### SIR_simtest

# Plots a comparison of susceptible and infected populations 

# Uses sir_sim (below)

# User specifies values for 4 parameters:
#   R0 is the basic reproductive number
#   gamma is the recovery rate parameter
#   I0 is the initial infected fraction
#   V is the initial immune fraction

# Populations are relative to total population N=1

# by Glenn Ledder
# written 2020/07/14
# revised 2020/11/08

# direct comments to gledder@unl.edu

### sir_sim_test appears below!

#########################################

sir_sim <- function(beta,gamma,I0,V,target)
	{

	## DATA

	maxdays = 1000

	## FUNCTION FOR THE DIFFERENTIAL EQUATION

    	yprime <- function(yy)
		{
      	S = yy[1]
      	I = yy[2]
      	Sp = -beta*S*I
      	Ip = -Sp-gamma*I
      	yp = c(Sp,Ip)
		return(yp)
		}

	## INITIALIZATION

	# set up results data structure with Y=[S,I] and R

	results = matrix(0,nrow=maxdays+1,ncol=3)	
	S0 = 1-I0-V
	Y = c(S0,I0)
	R = V
	results[1,] = c(Y,R)

	oldI = I0

	## FUNCTION FOR rk4

      # y0 is a column vector of initial conditions at t
      # yy is a column vector of values at t+dt

	rk4 <- function(dt,y0)
		{
		k1 = yprime(y0)
        	k2 = yprime(y0+0.5*dt*k1)
        	k3 = yprime(y0+0.5*dt*k2)
        	k4 = yprime(y0+dt*k3)
        	yy = y0+dt*(k1+2*k2+2*k3+k4)/6
		return(yy)
		}

	## COMPUTATION

	for (t in 1:maxdays)
		{
		Y = rk4(1,Y)
		R = 1-sum(Y)
    		results[t+1,] = c(Y,R)
    		if (Y[2]>min(target,oldI))
			{
        		oldI = Y[2]
			} else {
	      	results = results[1:(t+1),]
        		break
			}
		}

	## OUTPUT

	return(results)

	## END

	}

#########################################

### sir_sim_test starts here! ###

## SCENARIO DATA

R0 = 2.5
T = 5
I0 = 0.001
V = 0

## COMMON DATA

target = 0.001

## INITIALIZATION

par(mfrow=c(1,1))    # set the plotting area into a 1x1 array

## COMPUTATION

gamma = 1/T
beta = gamma*R0
results = sir_sim(beta,gamma,I0,V,target)
S = results[,1]
I = results[,2]
R = results[,3]
days = length(I)-1
new = c(0,S[1:days]-S[2:length(S)])

## OUTPUT

times = 0:days

plot(times, S, type="l", col="green3", ylim=c(0,1), xlab="days", ylab="population fraction", lwd=2, xaxs="i", yaxs="i")
lines(times, I, col="red", lwd=2)
lines(times, R, col="blue", lwd=2)
legend(50, 0.8, legend=c("S","I","R"),col=c("green3","red","blue"), lty=1, lwd=2)

cbind(times,S,I,R,new)
	
# maximum I
max(I)
	
# final S
S[days+1]

